# pylint: disable=line-too-long
"""Module implements The generic Blob Client for Azure

:raises NotImplementedError: Raised for when trying to upload a blob not of type BlockBlob
"""

from os import remove
from typing import ContextManager, Iterator, List

from datetime import datetime, timedelta
import logging

from azure.storage.blob import (
    BlobServiceClient,
    generate_account_sas,
    ResourceTypes,
    AccountSasPermissions,
    ContainerClient,
    BlobType,
)
from azure.core.exceptions import ClientAuthenticationError

LOGGER = logging.getLogger()


class AzureBlobClient:
    """Generic Blob Client for Azure"""

    def __init__(self, storage_account: str, credentials: str):
        """Builder of AzueBlobClient

        :param storage_account: Parameter from Azure: the storage account
        :type storage_account: str
        :param credentials: Parameter from Azure: the key (credential)
        :type credentials: str
        """
        self.storage_account = storage_account
        self.credentials = credentials
        self.account_url = "https://{account_name}.blob.core.windows.net".format(account_name=self.storage_account)
        self.sas_token = None

    def _generate_sas_signature(self, validity_in_secs: int = 30) -> str:
        """Generate an adapted SAS token

        :param validity_in_secs: Validity for the token expressed in seconds, defaults to 30
        :type validity_in_secs: int, optional
        :return: SAS token
        :rtype: str
        """
        return generate_account_sas(
            account_name=self.storage_account,
            account_key=self.credentials,
            resource_types=ResourceTypes(service=True, container=True, object=True),
            permission=AccountSasPermissions(read=True, list=True, create=True, add=True, remove=True, delete=True),
            expiry=datetime.utcnow() + timedelta(seconds=validity_in_secs),
        )

    def blob_container_client(self, container_name: str) -> ContainerClient:
        """Builder of container client

        :param container_name: Name of the container
        :type container_name: str
        :return: Container Client with goog authentication
        :rtype: ContainerClient
        """
        return ContainerClient(
            account_url=self.account_url, container_name=container_name, credential=self._generate_sas_signature()
        )

    def blob_service_client(self) -> BlobServiceClient:
        """Builder of service client

        :return: Service Client with goog authentication
        :rtype: BlobServiceClient
        """
        return BlobServiceClient(self.account_url, credential=self._generate_sas_signature())

    def list_containers(self) -> list:
        """Function to retrieve the containers

        :return: containers
        :rtype: list
        """
        containers = []
        for container in self.blob_service_client().list_containers():
            containers.append(container)
        return containers

    def list_blobs_in_container(self, container_name: str) -> list:
        """Function to list the blobs present in a container

        :param container: Name of the container to list
        :type container: str
        :return: list of the blobs
        :rtype: list
        """
        blobs = []
        for blob in self.blob_container_client(container_name=container_name).list_blobs():
            blobs.append(blob)
        return blobs

    def create_container(self, container_name: str) -> dict:
        """Create a container within the account storage

        :param container_name: Name of the container to create
        :type container_name: str
        :return: a description of the created container
        :rtype: dict
        """
        return self.blob_service_client().create_container(name=container_name)

    def delete_container(self, container_name: str):
        """Function to remove a container within the account storage

        :param container_name: Name of the container to remove
        :type container_name: str
        """
        return self.blob_service_client().delete_container(container=container_name)

    def upload_blob_in_container(
        self, container_name: str, blob_name: str, blob_filename: str, blob_type: BlobType = BlobType.BlockBlob
    ):
        """Upload a new blob in the container

        :param container_name: Name of the container in which we need to send the new blob
        :type container_name: str
        :param blob_name: Name to give to the new blob in the container
        :type blob_name: str
        :param blob_content: The blob content
        :type blob_content: bytes
        """
        if blob_type != BlobType.BlockBlob:
            LOGGER.error("Only the type 'BlockBlob' is supported for the moment")
            raise NotImplementedError
        with open(blob_filename, "rb") as data:
            self.blob_container_client(container_name=container_name).upload_blob(
                name=blob_name, data=data, blob_type=blob_type
            )

    def delete_blob_in_container(self, container_name: str, blob_name: str):
        """Function to delete a blob within a container

        :param container_name: Name of the container
        :type container_name: str
        :param blob_name: Name of the blob to delete
        :type blob_name: str
        """
        self.blob_container_client(container_name=container_name).delete_blob(blob=blob_name)